(* ::Package:: *)
(* Author: Antonio Romano / Roberto Cavaliere *)
(* This Mathematica application is part of the book 
	Geometric Optics
	Theory and Design of Astronomical Optical systems using Mathematica
	A. Romano, and R. Cavaliere
	May 11, 2016
*)

BeginPackage["GeometricOptics`"];

(******* Global Variables *******)
$MainDirectory::usage = "$MainDirectory gives the directory in which GeometricOptic application is installed";
$UserDefinedBaseDirectory::usage = "$UserDefinedBaseDirectory gives the main folder when the user installed the application in a non standard folder";
$IconsDirectory::usage = "$IconsDirectory gives the directory of icons used in the palette";
$PackagesDirectory::usage = "$PackagesDirectory gives the directory in which are loaded all modules of GeometricOptics";
$GeometricOpticsPackagesList::usage = "$GeometricOpticsPackagesList gives the full list of GeometricOptics packages available with the book";
$InstalledPackages::usage = "$InstalledPackages gives the list of GeometricOptics packages installed";
$PackageNames::usage = "$PackageNames gives the list of GeometricOptics packages full name";
$NotInstalledPackages::usage = "$NotInstalledPackages gives the list of GeometricOptics packages not installed";
$ExamplesArchive::usage = "$ExamplesArchive gives the list of all saved examples (one Dataset for all ind of projects)";
$ArgumentTypes::usage = "$ArgumentTypes gives the list of types of argument and related initialization value, for all functions (all packages) in GeometricOptic";
$SimplifyTimeLimit::usage="$SimplifyTimeLimit defines the maximum number of seconds to wait for Simplify trying to simplify symbolic expression";


(* $ExamplesStack: {PackageName, ExampleID, {parameters as rules name->value}} *)
$ExamplesStack::usage = "$ExamplesStack is the stack of Example executed in the curreent session.";

(* package-specific global variables *)
(* Replacede by ReportTemplate variable used by all packages 
	$TemplatesDirectory::usage = "$TemplatesDirectory[PackageName] gives the directory where are saved reports' template related to the package name given as argument"; *)
$ReportTemplate::usage = "$ReportTemplate is the general report template used by all packages to generate report";
$PrintTemplate::usage = "$PrintTemplate is the general print template used by all packages to print the table of results";
$BasicTemplate::usage = "$BasicTemplate is the general template of the list of input/output values used by all packages to return the calculation";

$ExamplesDirectory::usage = "$ExamplesDirectory[PackageName] gives the directory where are saved examples of projects related to the package name given as argument.";

(* variables used by packages *)
f1::usage"";
ft::usage"";
em::usage"";
alfa::usage"";
Ni::usage"";
beta::usage="";
delta::usage"";
capitaldelta::usage"";
gamma::usage"";
diam::usage"";
ind::usage"";
thick::usage"";
theta::usage"";
spes::usage="";
waves::usage="";
di::usage="";
dcp::usage="";
sigma::usage="";
\[Lambda]1::usage="";
\[Lambda]2::usage="";
\[Lambda]3::usage="";
rad::usage="";
stoprad::usage="";
nstop::usage="";
dis::usage="";
dobject::usage="";
hobject::usage="";
CameraType::usage="";
OutputType::usage="";
OutputLevel::usage="";


(* the following variables are exported to be used by user as working variables, they are generated by internal functions *)
GOn::usage = "";
GOwe::usage = "";
GOren::usage = "";
GOwen::usage = "";
GOrexit::usage = "";
GOdistancefp::usage = "";
GOdistancesp::usage = "";
GOdistancegauss::usage = "";
GOfocalLength::usage = "";
GOimageHeight::usage = "";
GOSphericalCoefficient = "";
GOComaCoefficient = "";
GOAstigmatismCoefficient = "";
GOaberration::usage = "";
GOcoma::usage = "";
GOastigmatism::usage = "";
GOcurvature::usage = "";
GOSeidelCurvature::usage = "";
GOPetzvalRadius::usage = "";
GOdistortion::usage = "";
GOaberrationexit::usage = "";
GOcomaexit::usage = "";
GOastigmatismexit::usage = "";
GOdistortionexit::usage = "";

GON::usage="";

(* variables used by palette *)
fileOpenerIcon::usage = "";
selectAllIcon::usage = "";
clearAllIcon::usage = "";
packageLoaderIcon::usage = "";
checkInstallationIcon::usage = "";
folderOpenIcon::usage = "";
mainIcon::usage = "";
helpIcon::usage = "";
saveLastIcon::usage = "";
saveFromListIcon::usage = "";


(******* Global Functions *******)
NameFromPackageName::usage = "NameFromPackageName[package] gives the print name for a package (adds blank space in the name of the package )";
						
Options[TotalAberrations] = {OutputType->None, OutputLevel->"Minimal"};
(* 	OutputType can be 
					"Report", generates a new report (notebook) for each call to the function
					"Print", prints results in the current position of the calling notebook
					"Basic", gives the list of results 
					"Values", gives the list of values (no names of variables)
					None or any other directive, no visible output returned, however output variable are set
	OutputLevel can be
					"Minimal", gives only output values, those calculated by the function
					"Full", gives the list of input parameters and output values
*)
						
TotalAberrations::usage = "TotalAberrations is a common function for all packages";

CheckInstalledPackages::usage = "CheckInstalledPackages[] gives the list of installed packages of GeometricOptics";
LoadPackage::usage = "LoadPackage[name] loads the package name, if it is installed";
GenerateMainPalette::usage = "GenerateMainPalette creates the Geometric Optics main palette";

ArgumentsQ::usage="ArgumentsQ";

DeleteExampleDuplicates::usage = "DeleteExampleDuplicates[ExampleList] clears duplicates in an example list";
SaveExample::usage = "SaveExample[package] saves a default example for the given package or uses the argument if provided as second argument (optional)";
SaveLastExample::usage="SaveLastExample[] saves the latest computed example (the last entry in $ExamplesStack)";
SaveExampleFromList::usage="SaveExampleFromList[] lets the user selec an example to save, from the list of current session's examples";
LoadExample::usage = "LoadExample[package] allows to load an example from the archives";

Begin["`Private`"];

(* Note: global variable have to be defined before global functions, otherwise function may not recognise variables *)
   
(******* Global Variables *******)
$SimplifyTimeLimit = 5;

$MainDirectory = Which[
						DirectoryQ[FileNameJoin[{$UserBaseDirectory, "Applications", "GeometricOptics"}]],
						FileNameJoin[{$UserBaseDirectory, "Applications", "GeometricOptics"}],
						
						DirectoryQ[FileNameJoin[{$BaseDirectory, "Applications", "GeometricOptics"}]],
						FileNameJoin[{$BaseDirectory, "Applications", "GeometricOptics"}],
						
						ValueQ[$UserDefinedBaseDirectory],
						$UserDefinedBaseDirectory,
												
						True,
						(* the package is installed in a non standard position, the first time the application is loaded requires to track the path *)
						$UserDefinedBaseDirectory = DialogInput[{temp = ""},
																Column[{"Specify the path where GeometricOptics is installed",
																		Column[{"NOTE: this operation will add the following two lines of code", Spacer[5],
																				Style["AppendTo[$Path, \"yourpath\"];\n$UserDefinedBaseDirectory = \"yourpath\";", Bold],
																				"to current user's init.m file located in",
																				Style[$UserBaseDirectory, Bold], Spacer[5],
																				"This is needed to allow GeometricOptic package to run from a non-standard position."}],
																		Column[{InputField[Dynamic[temp], String, FieldHint -> "Enter the path or use Browse to select the installation folder", FieldSize -> 40], 
																				FileNameSetter[Dynamic[temp], "Directory"],
																				Row[{DefaultButton[DialogReturn[temp]], CancelButton[DialogReturn[]]}]}, Alignment -> Center, Spacings -> 2], Spacer[5]}, 
																		Spacings -> 2, Alignment -> Left], 
																WindowTitle -> "Set GeometricOptics installation path to a non-standard position"];
						If[$UserDefinedBaseDirectory === "", 
						   MessageDialog["Installation path has not been assigned, the application may not work properly.", WindowTitle->"Warning: operation not completed"];
						   Clear[$UserDefinedBaseDirectory],
						   (* the path has been defined, being the first time it is added to $Path and the $UserBaseDirectory init.m is modified *)
						   	Export[FileNameJoin[{$UserBaseDirectory, "Kernel", "init.m"}],
								   Import[FileNameJoin[{$UserBaseDirectory, "Kernel", "init.m"}], "Text"] <> "\nAppendTo[$Path,\"" <> temp <> "\"];\n$UserDefinedBaseDirectory=\"" <> temp <> "\";", "Text"];
							$UserDefinedBaseDirectory]];



$PackagesDirectory = FileNameJoin[{$MainDirectory, "Packages"}];

$IconsDirectory = FileNameJoin[{$MainDirectory, "Icons"}];

(* $TemplatesDirectory[packagename_String] = FileNameJoin[{$PackagesDirectory, packagename, "Templates"}]; *)

$ReportTemplate = Notebook[{
							Cell[TemplateSlot["title"], "Title"],
							Cell[TextData[{TemplateExpression[DateString[]]}], "Date"],
							Cell[TextData[{TemplateSlot["function"]}], "Function"],
							TemplateExpression@Apply[Sequence, TemplateIf[TemplateSlot["outputlevel"] === "Full", 
																			(* input and output tables *)
																			{Cell["Input parameters", "InputParameterTop"], Cell[BoxData[ToBoxes[TemplateSlot["inPanel"]]], "InputParameterBottom"],
																			 Cell["Output values", "OutputValueTop"], Cell[BoxData[ToBoxes[TemplateSlot["outPanel"]]], "OutputValueBottom"]},
																			TemplateIf[TemplateSlot["outputlevel"] === "Coefficients",
																						{Cell["Output values (aberration coefficients)", "OutputValueTop"], 
																						 Cell[BoxData[ToBoxes[TemplateSlot["outPanel"]]], "OutputValueBottom"]},
																						(* only output table *)
																						{Cell["Output values", "OutputValueTop"], 
																						 Cell[BoxData[ToBoxes[TemplateSlot["outPanel"]]], "OutputValueBottom"]}]]]
							},
							(* Notebook Options *)
							ShowStringCharacters -> False,
							WindowTitle -> TemplateExpression[TemplateSlot["function"] <> " report"],
							WindowSize -> {Scaled[1/2], Scaled[4/5]},
							WindowMargins -> Automatic,
							ScreenStyleEnvironment -> "Report",
							StyleDefinitions -> "GeometricOptics.nb"];

$PrintTemplate = Panel[Column[{
								Style[TemplateSlot["title"], "Title"],
								Style[TemplateExpression[DateString[]], "Date"],
								Style[TemplateSlot["function"], "Function"],
								Spacer[10],
								TemplateExpression@Apply[Sequence, TemplateIf[TemplateSlot["outputlevel"] === "Full",
																				(*input and output tables*)
																				{Style["Input parameters", "InputParameterTop"], 
																				 TemplateSlot["inPanel"], 
																				 Spacer[5],
																				 Style["Output values", "OutputValueTop"], 
																				 TemplateSlot["outPanel"]},
																				 
																				 (*only output table*)
																				 {Style["Output values", "OutputValueTop"],
																				  TemplateSlot["outPanel"]}]]}], 
						BaseStyle -> "InputParameterTop", 
						Background -> GrayLevel[0.9]];
						
$BasicTemplate = TemplateExpression@TemplateIf[TemplateSlot["outputlevel"] === "Full",
												(*input and output values *)
												{{"Input parameters", TemplateSlot["inputs"]},
												 {"Output values", TemplateSlot["outputs"]}},
												{"Output values", TemplateSlot["outputs"]}];
																 
$ExamplesDirectory[packagename_String] = FileNameJoin[{$PackagesDirectory, packagename, "Examples"}];
(* if the user load the package more than one time during a Mathematica session, it has to keep the stack *)
$ExamplesStack = If[ValueQ[$ExamplesStack], $ExamplesStack, {}];

$ArgumentTypes = Dataset[{	(* the argument can be only a number *)
							<|"Type"->"Number", "DefaultValue"->0, "CheckFunction" -> (NumericQ[#] &), "Description"->"Numerical argument"|>,
							(* tha argument can be a number or a symbol - for symbolic computations of optical system design *)
							<|"Type"->"Symbolic", "DefaultValue"->0, "CheckFunction" -> (Head[#] == DirectedInfinity || AtomQ[#] &), "Description"->"Numeric or symbolic argument"|>,
							(* the argument can be a list of numeric or symbolic values *)
							<|"Type"->"List", "DefaultValue"->{}, "CheckFunction" -> (VectorQ[#, (Head[#] === DirectedInfinity || AtomQ[#] &)] &), "Description"->"List of numeric or symbolic arguments"|>,
							(* the argument can be a list of list with numeric or symbolic values *)
							<|"Type"->"ListOfList", "DefaultValue"->{{}}, "CheckFunction" -> (MatrixQ[#, AtomQ] &), "Description"->"List of list of numeric or symbolic arguments"|>
						 }];

(* $GeometricOpticsPackagesList defines features of main function defined in each package *)
$GeometricOpticsPackagesList = Dataset[{<|"PackageID" ->  1, "Category" -> "Camera", 	"PackageName" -> "BakerSchmidtCameraCP", "Description" -> "Baker-Schmidt Camera (conic primary)",
										  "Arguments" -> <|	"f1" 		-> <|"Type"->"Number", "Description"->"Primary focal"|>,
															"\[Gamma]" 	-> <|"Type"->"Number", "Description"->"-\[Gamma]*f1 is the radius of secondary mirror"|>,
															"\[Beta]" 	-> <|"Type"->"Number", "Description"->"1-\[Alpha], where \[Alpha] is the obstruction factor (\[Alpha] \[GreaterEqual] 0.40)"|>,
															"diam" 		-> <|"Type"->"Number", "Description"->"Diameter of the primary mirror"|>, 
															"\[Theta]" 	-> <|"Type"->"Number", "Description"->"Field angle in degrees"|>
														  |>
										 |>,
										<|"PackageID" ->  2, "Category" -> "Camera", 	"PackageName" -> "BakerSchmidtCameraSM", "Description" -> "Baker-Schmidt Camera (spherical mirrors)",
										  "Arguments" -> <|	"f1" 		-> <|"Type"->"Number", "Description"->"Primary focal"|>,
															"\[Beta]" 	-> <|"Type"->"Number", "Description"->"1-\[Alpha], where \[Alpha] is the obstruction factor (\[Alpha] \[GreaterEqual] 0.40)"|>,
															"diam" 		-> <|"Type"->"Number", "Description"->"Diameter of the primary mirror"|>, 
															"\[Theta]" 	-> <|"Type"->"Number", "Description"->"Field angle in degrees"|>,
															"sf" 		-> <|"Type"->"Number", "Description"->"TOBEDEFINED"|>
														  |>
										 |>,  
										<|"PackageID" ->  3, "Category" -> "Camera", 	"PackageName" -> "BuchroederCamera", "Description" -> "Buchroeder Camera",
										  "Arguments" -> <|	"f1" 		-> <|"Type"->"Number", "Description"->"Focal length of the primary mirror"|>,
															"spes" 		-> <|"Type"->"List", "Description"->"List of thicknesses of the lenses"|>,
															"diam" 		-> <|"Type"->"Number", "Description"->"Diameter of the primary mirror"|>, 
															"Ni" 		-> <|"Type"->"Number", "Description"->"Refractive index of the lens"|>, 
															"\[Theta]" 	-> <|"Type"->"Number", "Description"->"Field angle in degrees"|>
														  |>
										 |>,
										<|"PackageID" ->  4, "Category" -> "Camera", 	"PackageName" -> "WrightCamera", "Description" -> "Wright's Camera",
										  "Arguments" -> <|	"f1" 				-> <|"Type"->"Number", 	"Description"->"Primary focal length"|>,
															"Ni" 				-> <|"Type"->"Number", 	"Description"->"Refractive index of the corrector"|>,
															"thick" 			-> <|"Type"->"Number", 	"Description"->"Thickness of the corrector"|>,
															"diam"				-> <|"Type"->"Number", 	"Description"->"Diameter of corrector"|>,
															"\[CapitalDelta]"	-> <|"Type"->"Number", 	"Description"->"Distance between the corrector and the mirror"|>,
															"\[Theta]" 	-> <|"Type"->"Number", 	"Description"->"Field angle in degrees"|>
														  |>										
										 |>,

										<|"PackageID" ->  5, "Category" -> "Camera", 	"PackageName" -> "HoughtonCamera", "Description" -> "Houghton's Camera",
										  "Arguments" -> <|	"f1" 				-> <|"Type"->"Number", 	"Description"->"Focal length of the mirror"|>,
															"spes" 				-> <|"Type"->"List", 	"Description"->"List of thicknesses of the lenses"|>,
															"diam" 				-> <|"Type"->"Number", 	"Description"->"Diameter of the corrector"|>,
															"\[CapitalDelta]" 	-> <|"Type"->"Number", 	"Description"->"Distance between the corrector and the mirror"|>,
															"Ni" 				-> <|"Type"->"Number", 	"Description"->"Refractive index of the corrector"|>,
															"\[Theta]" 			-> <|"Type"->"Number", 	"Description"->"Field angle in degrees"|>
														  |>
										 |>,

										<|"PackageID" ->  6, "Category" -> "Camera", 	"PackageName" -> "SchmidtCamera", "Description" -> "Schmidt's Camera",
										  "Arguments" -> <|	"f1" 		-> <|"Type"->"Number", 		"Description"->"Focal length of the mirror"|>,
															"ind" 		-> <|"Type"->"ListOfList", 	"Description"->"List of the refractive indices"|>,
															"thick" 	-> <|"Type"->"Number", 		"Description"->"Thickness of the corrector"|>,
															"diam" 		-> <|"Type"->"Number", 		"Description"->"Diameter of the corrector"|>,
															"\[Alpha]" 	-> <|"Type"->"Number", 		"Description"->"Form factor of the second surface of the corrector"|>,
															"\[Theta]" 	-> <|"Type"->"Number", 		"Description"->"Field angle in degrees"|>
														  |>
										 |>,

										<|"PackageID" ->  7, "Category" -> "Camera", 	"PackageName" -> "MaksutovCamera", "Description" -> "Maksutov's Camera",
										  "Arguments" -> <|	"f1" 		-> <|"Type"->"Number", 	"Description"->"Focal length of the mirror"|>,
															"diam" 		-> <|"Type"->"Number", 	"Description"->"Diameter of the corrector"|>,
															"\[Theta]" 	-> <|"Type"->"Number", 	"Description"->"View angle in degrees"|>
														  |>										
										 |>,
										
										<|"PackageID" ->  8, "Category" -> "Telescope", "PackageName" -> "HoughtonCassegrain", "Description" -> "Houghton-Cassegrain Telescope",
										  "Arguments" -> <|	"f1" 		-> <|"Type"->"Number", 	"Description"->"Primary focal length"|>,
															"ft" 		-> <|"Type"->"Number", 	"Description"->"Total focal length"|>,
															"spes" 		-> <|"Type"->"List", 	"Description"->"List of thicknesses of the lenses"|>,
															"em" 		-> <|"Type"->"Number", 	"Description"->"Back focal distance"|>, 
															"diam" 		-> <|"Type"->"Number", 	"Description"->"Diameter of the corrector"|>,
															"\[Delta]" 	-> <|"Type"->"Number", 	"Description"->"Distance of the doublet from mirror"|>,
															"Ni" 		-> <|"Type"->"Number", 	"Description"->"Refractive index of the corrector"|>,
															"\[Theta]" 	-> <|"Type"->"Number", 	"Description"->"Field angle in degrees"|>
														  |>
										 |>,
										
										<|"PackageID" ->  9, "Category" -> "Telescope", "PackageName" -> "KlevtsovTelescope", "Description" -> "Klevtsov combination",
										  "Arguments" -> <|	"f1" 		-> <|"Type"->"Number", 	"Description"->"Principal mirror focal length"|>,
															"ft" 		-> <|"Type"->"Number", 	"Description"->"Total focal length"|>,
															"em" 		-> <|"Type"->"Number", 	"Description"->"Back focal distance"|>,
															"diam" 		-> <|"Type"->"Number", 	"Description"->"Diameter of the principal mirror"|>,
															"\[Theta]" 	-> <|"Type"->"Number", 	"Description"->"View angle in degrees"|>
														  |>
										 |>,

										<|"PackageID" ->  10, "Category" -> "Telescope", "PackageName" -> "MaksutovCassegrain",	"Description" -> "Maksutov-Cassegrain Telescope",
										  "Arguments" -> <|	"f1" 		-> <|"Type"->"Number", 	"Description"->"Primary focal length"|>,
															"ft" 		-> <|"Type"->"Number", 	"Description"->"Total focal length"|>,
															"em" 		-> <|"Type"->"Number", 	"Description"->"Back focal distance"|>, 
															"\[Delta]" 	-> <|"Type"->"Number", 	"Description"->"Distance of doublet from mirror"|>,
															"diam" 		-> <|"Type"->"Number", 	"Description"->"Diameter of the corrector"|>,
															"\[Theta]" 	-> <|"Type"->"Number", 	"Description"->"Field angle in degrees"|>
														  |>
										
										 |>,
										
										<|"PackageID" -> 11, "Category" -> "Telescope", "PackageName" -> "HoughtonCassegrainC", "Description" -> "Houghton's Cassegrain Telescope with corrector at the prime focus",
										  "Arguments" -> <|	"f1" 		-> <|"Type"->"Number", 		"Description"->"Primary focal length"|>,
															"ft" 		-> <|"Type"->"Number", 		"Description"->"Total focal length"|>,
															"ind" 		-> <|"Type"->"ListOfList", 	"Description"->"List of the refractive indices"|>,
															"em" 		-> <|"Type"->"Number", 		"Description"->"Back focal distance"|>, 
															"\[Sigma]"	-> <|"Type"->"Number",  	"Description"->"Fraction of em in the first step of the project (Cassegrain with conic primary)"|>,
															"diam" 		-> <|"Type"->"Number", 		"Description"->"Diameter of the corrector"|>,
															"\[Theta]" 	-> <|"Type"->"Number", 		"Description"->"Field angle in degrees"|>,
															"waves" 	-> <|"Type"->"List", 		"Description"->"The wave-length corresponding to the chosen refractive indices"|>,
															"dcp" 		-> <|"Type"->"Number", 		"Description"->"Distance between the first surface of the corrector and the primary"|>,
															"thick" 	-> <|"Type"->"List", 		"Description"->"Thickness of corrector"|>,
															"di" 		-> <|"Type"->"Number", 		"Description"->"Distance between lenses"|>															
														  |>
										 |>,
										
										<|"PackageID" -> 12, "Category" -> "Telescope", "PackageName" -> "SchmidtCassegrainCBefore", "Description" -> "Schmidt-Cassegrain telescope with a corrector in front of the Cassegrain combination",
										  "Arguments" -> <|	"f1" 				-> <|"Type"->"Number", 	"Description"->"Primary focal length"|>,
															"ft" 				-> <|"Type"->"Number", 	"Description"->"Total focal length"|>,
															"em" 				-> <|"Type"->"Number", 	"Description"->"Back focal distance"|>, 
															"Ni" 				-> <|"Type"->"Number", 	"Description"->"Refractive index of the corrector"|>,
															"thick" 			-> <|"Type"->"Number", 	"Description"->"Thickness of corrector"|>,
															"diam" 				-> <|"Type"->"Number", 	"Description"->"Primary diameter"|>,
															"\[Delta]" 			-> <|"Type"->"Number", 	"Description"->"Fraction of distance corrector-primary in focal unity"|>,
															"\[Theta]" 			-> <|"Type"->"Number", 	"Description"->"Field angle in degrees"|>
														  |>
										 |>,

										<|"PackageID" -> 13, "Category" -> "Telescope", "PackageName" -> "CassegrainCombination", "Description" -> "Cassegrain Telescope combination",
										  "Arguments" -> <|	"f1" 		-> <|"Type"->"Number", 	"Description"->"Primary focal length"|>,
															"ft" 		-> <|"Type"->"Number", 	"Description"->"Total focal length"|>,
															"em" 		-> <|"Type"->"Number", 	"Description"->"Back focal distance"|>, 
															"diam" 		-> <|"Type"->"Number", 	"Description"->"Primary diameter"|>,
															"\[Theta]" 	-> <|"Type"->"Number", 	"Description"->"Field angle in degrees"|>
														  |>
										 |>,
										
										<|"PackageID" -> 14, "Category" -> "Telescope", "PackageName" -> "SchmidtCassegrainCAfter", "Description" -> "Schmidt-Cassegrain Telescope (corrector after Cassegrain combination)",
										  "Arguments" -> <|	"f1" 				-> <|"Type"->"Number", 	"Description"->"Primary focal length"|>,
															"ft" 				-> <|"Type"->"Number", 	"Description"->"Total focal length"|>,
															"em" 				-> <|"Type"->"Number", 	"Description"->"Back focal distance"|>, 
															"diam" 				-> <|"Type"->"Number", 	"Description"->"Primary diameter"|>,
															"\[Theta]" 			-> <|"Type"->"Number", 	"Description"->"Field angle in degrees"|>
														  |>
										 |>,

										<|"PackageID" -> 15, "Category" -> "Telescope", "PackageName" -> "SchmidtCassegrainCBeforeSM", "Description" -> "Schmidt's Cassegrain Telescope (corrector after Cassegrain combination and spherical mirrors)",
										  "Arguments" -> <|	"f1" 				-> <|"Type"->"Number", 	"Description"->"Primary focal length"|>,
															"ft" 				-> <|"Type"->"Number", 	"Description"->"Total focal length"|>,
															"em" 				-> <|"Type"->"Number", 	"Description"->"Back focal distance"|>, 
															"diam" 				-> <|"Type"->"Number", 	"Description"->"Primary diameter"|>,
															"Ni" 				-> <|"Type"->"Number", 	"Description"->"Refractive index of the corrector"|>,
															"\[CapitalDelta]" 	-> <|"Type"->"Number", 	"Description"->"Thickness of corrector"|>,
															"\[Theta]" 			-> <|"Type"->"Number", 	"Description"->"Field angle in degrees"|>
														  |>
										 |>,										 
										 
										<|"PackageID" -> 16, "Category" -> "Telescope", "PackageName" -> "Fraunhofer", 	"Description" -> "Fraunhofer's doublet",
										  "Arguments" -> <|	"thick" 	-> <|"Type"->"List", 		"Description"->"Thickness of lenses"|>,
															"ind" 		-> <|"Type"->"ListOfList", 	"Description"->"List of the refractive indices"|>,
															"diam" 		-> <|"Type"->"Number", 		"Description"->"Primary diameter"|>,
															"\[Theta]" 	-> <|"Type"->"Number", 		"Description"->"Field angle in degrees"|>,
															"waves" 	-> <|"Type"->"List", 		"Description"->"The wave-length corresponding to the chosen refractive indices"|>,
															"focal" 	-> <|"Type"->"Number", 		"Description"->"Focal"|>
														  |>
										 |>,

										<|"PackageID" -> 17, "Category" -> "Telescope", "PackageName" -> "Triplet", "Description" -> "Triplet",
										  "Arguments" -> <|	"thick" 	-> <|"Type"->"List", 		"Description"->"List of the distances between the surfaces along the optical axis"|>,
															"ind" 		-> <|"Type"->"ListOfList", 	"Description"->"List of the refractive indices"|>,
															"diam" 		-> <|"Type"->"Number", 		"Description"->"Primary diameter"|>,
															"\[Theta]" 	-> <|"Type"->"Symbolic", 	"Description"->"The field angle, if the object is at infinity"|>,
															"waves" 	-> <|"Type"->"List", 		"Description"->"The wave-length corresponding to the chosen refractive indices"|>,
															"focal" 	-> <|"Type"->"Number", 		"Description"->"Focal"|>
														  |>
										 |>,

(* this package are not included in this application distribution  
										<|"PackageID" -> 18, "Category" -> "General", 	"PackageName" -> "GaussianData", "Description" -> "Gaussian Data of a Centered Optical System",
										  "Arguments" -> <|	"rad" 		-> <|"Type"->"List", 		"Description"->"List of the radii of the surfaces"|>,
															"thick" 	-> <|"Type"->"List", 		"Description"->"List of the distances between the surfaces along the optical axis"|>,
															"ind" 		-> <|"Type"->"List", 		"Description"->"List of the refractive indices"|>,
															"stoprad"	-> <|"Type"->"Number", 		"Description"->"Radius of the aperture stop"|>,
															"nstop" 	-> <|"Type"->"Number", 		"Description"->"Number of the surfaces before the aperture stop"|>,
															"dis" 		-> <|"Type"->"Number", 		"Description"->"if nstop = 0, then dis is the (negative) distance of the aperture stop from the first surface after it. 
																											   If nstop >0, then dis denotes the distance between the last surface before the aperture stop and the aperture stop"|>,
															"dobject" 	-> <|"Type"->"Symbolic", 	"Description"->"Distance of the object from the first surface"|>,
															"hobject" 	-> <|"Type"->"Symbolic", 	"Description"->"Height of the object if it is at a finite distance"|>,
															"\[Theta]" 	-> <|"Type"->"Symbolic", 	"Description"->"The field angle, if the object is at infinity"|>
														  |>
										 |>,
										
										<|"PackageID" -> 19, "Category" -> "General", 	"PackageName" -> "SingleRayTracing", "Description" -> "Single Ray Tracing (Polar Coordinates)",
										  "Arguments" -> <|	"rad" 		-> <|"Type"->"List", 		"Description"->"List of the radii of the surfaces"|>,
															"thick" 	-> <|"Type"->"List", 		"Description"->"List of the distances between the surfaces along the optical axis"|>,
															"ind" 		-> <|"Type"->"ListOfList", 	"Description"->"List of the refractive indices"|>,
															"costasf"	-> <|"Type"->"List", 		"Description"->"List of the constant which characterize the nature of any surface 
																												(0 for a sphere, K for a conic, {a4, as} for an aspheric surface)"|>,
															"stopradius"-> <|"Type"->"Number", 		"Description"->"Radius of the aperture stop"|>,
															"nstop" 	-> <|"Type"->"Number", 		"Description"->"Number of the surfaces before the aperture stop"|>,
															"dis" 		-> <|"Type"->"Number", 		"Description"->"if nstop = 0, then dis is the (negative) distance of the aperture stop from the first surface after it. 
																											   If nstop >0, then dis denotes the distance between the last surface before the aperture stop and the aperture stop"|>,
															"dobject" 	-> <|"Type"->"Symbolic", 	"Description"->"Distance of the object from the first surface"|>,
															"hobject" 	-> <|"Type"->"Number", 		"Description"->"Height of the object if it is at a finite distance"|>,
															"angle" 	-> <|"Type"->"Symbolic", 	"Description"->"The field angle, if the object is at infinity"|>,
															"waves" 	-> <|"Type"->"List", 		"Description"->"The wave-length corresponding to the chosen refractive indices"|>
														  |>										
										 |>,
*)
										
										<|"PackageID" -> 20, "Category" -> "General", 	"PackageName" -> "TotalAberrations", "Description" -> "Total Aberrations",
										  "Arguments" -> <|	"rad" 		-> <|"Type"->"List", 		"Description"->"List of the radii of the surfaces"|>,
															"thick" 	-> <|"Type"->"List", 		"Description"->"List of the distances between the surfaces along the optical axis"|>,
															"ind" 		-> <|"Type"->"ListOfList", 	"Description"->"List of the refractive indices"|>,
															"costasf"	-> <|"Type"->"List", 		"Description"->"List of the constant which characterize the nature of any surface 
																												(0 for a sphere, K for a conic, {a4, as} for an aspheric surface)"|>,
															"stopradius"-> <|"Type"->"Number", 		"Description"->"Radius of the aperture stop"|>,
															"nstop" 	-> <|"Type"->"Number", 		"Description"->"Number of the surfaces before the aperture stop"|>,
															"dis" 		-> <|"Type"->"Number", 		"Description"->"if nstop = 0, then dis is the (negative) distance of the aperture stop from the first surface after it. 
																											   If nstop >0, then dis denotes the distance between the last surface before the aperture stop and the aperture stop"|>,
															"dobject" 	-> <|"Type"->"Symbolic", 	"Description"->"Distance of the object from the first surface"|>,
															"hobject" 	-> <|"Type"->"Number", 		"Description"->"Height of the object if it is at a finite distance"|>,
															"angle" 	-> <|"Type"->"Symbolic", 	"Description"->"The field angle, if the object is at infinity"|>,
															"waves" 	-> <|"Type"->"List", 		"Description"->"The wave-length corresponding to the chosen refractive indices"|>
														  |>										
										 |>}];

$InstalledPackages = $GeometricOpticsPackagesList[Select[MemberQ[Map[FileBaseName, FileNames["*", {$PackagesDirectory}]], #PackageName] &]];

$PackageNames = Normal[$InstalledPackages[All, "PackageName"]];

$NotInstalledPackages = Complement[$GeometricOpticsPackagesList, $InstalledPackages];

$ExamplesArchive = Quiet@Dataset[DeleteCases[Flatten@Map[Import[FileNameJoin[{$ExamplesDirectory[#], "Examples.m"}]] &, $PackageNames], Null | $Failed]];
(* if none of the package's examples are found, set the example archive to the empty dataset *)
If[$ExamplesArchive === {}, $ExamplesArchive = Dataset[{}]];


(******* Global Functions *******)
(* NameFromPackageName[package_String] := StringRiffle[StringSplit[package, RegularExpression["(?=[[:upper:]])"]]]; *)
NameFromPackageName[package_String] := $GeometricOpticsPackagesList[SelectFirst[#PackageName == package &], "Description"];

TotalAberrations[rad_, thick_, ind_, costasf_, stoprad_, nstop_, dis_, dobject_, hobject_, angle_, waves_, OptionsPattern[]]:=
Quiet@Module[{n, m, rade, inde1, ze, ze1, de, ye, diame, ye1, zee, zee1, fe, yee, yee1, Mae, P, R0, Mt, D0, B0, b, zp, z1p, zog, zog1, test, zoginf, zog1inf, 
		foc, hobj1, C1, C2, C3, C4, C5, c1, d1, k1, Cd1, Cd2, Cd3, Cd4, Cd5, theta, mhobject, Absfer, Absftot, Comapar, Comatot, Asttot, Astpar, Sp, 
		Seidcurv, Curvpar, Curvtot, Petzval, Seidsphe, Seidcoma, Seidast, Seiddist, Disttot, Distpar, h}, 

		(* number of surfaces and wave lengths *)
		n = Length[rad];
		m = Length[waves];
		(* Evaluation of the aperture stop images supplied by the surfaces preceeding the stop:the last one is the entrance pupil. *)
		rade = -Reverse[Take[rad, nstop]];
		
		diame = ConstantArray[stoprad, m];
		
		If[nstop === 0, 
		   inde1 = ConstantArray[1, m];
		   ze = ConstantArray[dis, {m, 1}];
		   de = ConstantArray[dis, m], 
		 
		   inde1 = Map[Prepend[Reverse[#], 1]&, ind[[All, 1;;nstop]]];
		   ze = ConstantArray[-dis, {m, 1}];
		   ze1 = ConstantArray[0, {m, nstop}];
		   Table[ze1[[h, i]] = Which[
		                             ze[[h, i]] === 0, 
									 ze[[h, i]], 
									 ze[[h, i]] === -Infinity && Abs[rad[[i]]] === Infinity, 
									 -Infinity, 
									 ze[[h, i]] =!= -Infinity && Simplify[Numerator[Together[inde1[[h, i]]/ze[[h, i]] - inde1[[h, i]]/rad[[i]] + inde1[[h, i + 1]]/rad[[i]]]]] === 0, 
									 -Infinity, 
									 ze[[h, i]] =!= -Infinity || ze[[h, i]] === -Infinity, 
									 ReplaceAll[x, Flatten@Quiet@Solve[inde1[[h, i]]/ze[[h, i]] - inde1[[h, i]]/rade[[i]] - inde1[[h, i + 1]]/x + inde1[[h, i + 1]]/rade[[i]] == 0, x]]];
                 AppendTo[ze[[h]], ze1[[h, i]] - Append[thick, 0][[i]]], {h, m}, {i, nstop}];
			ye = ConstantArray[stoprad, {m, 1}];
			ye1= ConstantArray[stoprad, {m, nstop}];
			Table[ye1[[h, i]] = If[ze[[h, i]] =!= 0, (inde1[[h, i]]*ze1[[h, i]])/(inde1[[h, i + 1]]*ze[[h, i]])*ye[[h, i]], ye[[h, i]]];
				  AppendTo[ye[[h]], ye1[[h, i]]], {h, m}, {i, nstop}];
			diame = ye1[[All, nstop]];
			de = ze1[[All, nstop]];
		   ];
 
		zee = Partition[de, 1];
		zee1 = ConstantArray[0, {m, n}];
		Table[zee1[[h, i]] = Which[zee[[h, i]] === 0, 
								   0,
								   zee[[h, i]] === -Infinity && Abs[rad[[i]]] === Infinity, 
								   -Infinity, 
								   zee[[h, i]] =!= -Infinity && ind[[h, i]]/zee[[h, i]] - ind[[h, i]]/rad[[i]] + ind[[h, i + 1]]/rad[[i]] === 0, 
								   -Infinity, 
								   zee[[h, i]] =!= -Infinity || zee[[h, i]] === -Infinity, 
								   ReplaceAll[x, Flatten@Quiet@Solve[Numerator[Together[ind[[h, i]]/zee[[h, i]] - ind[[h, i]]/rad[[i]] - ind[[h, i + 1]]/x + ind[[h, i + 1]]/rad[[i]]]] == 0, x]]];
			  AppendTo[zee[[h]], zee1[[h, i]] - Append[thick, 0][[i]]], {i, n}, {h, m}];
		yee = ConstantArray[0, {m, n+1}];
		yee[[All, 1]] = diame;
		yee1 = ConstantArray[0, {m, n}];
		Table[yee1[[h, i]] = If[zee[[h, i]] =!= 0, (ind[[h, i]] * zee1[[h, i]])/(ind[[h, i + 1]] * zee[[h, i]]) * yee[[h, i]], yee[[h, i]]];
			  yee[[h, i + 1]] = yee1[[h, i]], {i, n}, {h, m}];
	 	(* Pupil magnification *)
		Mae = yee1/yee[[All, 1]];
		(* Principal planes and focal length *)
		P = Table[(ind[[h, i + 1]] - ind[[h, i]])/rad[[i]], {h, m}, {i, n}];
		R0 = Table[{{1, 0}, {-P[[h, i]]/ind[[h, i + 1]], ind[[h, i]]/ind[[h, i + 1]]}}, {h, m}, {i, n}];
		If[n === 1, 
		   Mt = R0[[All, 1]], 
		   (* Total matrix of the i-th surface *)
		   D0 = Table[{{1, thick[[i]]}, {0, 1}}, {i, n-1}];
		   B0 = Table[If[i < n, D0[[i]].R0[[h, i]], R0[[h, n]]], {h, m}, {i, n}];
		   (* matrices of the sub-systems *)
		   b = Partition[B0[[All, 1]], 1];
		   Table[AppendTo[b[[h]], B0[[h, i]].b[[h, i - 1]]], {h, m}, {i, 2, n}];
		   (* total matrix of the system *)
		   Mt = b[[All, n]]];
		
		(* Distances of the principal planes from the first and last surface, respectively *)
		{zp, z1p} = Transpose@Table[If[Mt[[h, 2, 1]] =!= 0, 
									   {(Mt[[h, 1, 2]] * Mt[[h, 2, 1]] + Mt[[h, 2, 2]] - Mt[[h, 1, 1]] * Mt[[h, 2, 2]])/Mt[[h, 2, 1]], (1 - Mt[[h, 1, 1]])/Mt[[h, 2, 1]]}, 
									   {0, 0}], {h, m}];
		
		(* Distances of the images *)
		zog = ConstantArray[dobject, {m, 1}];
		zog1 = ConstantArray[0, {m, n}];
		Table[
		      test = Quiet@Simplify[(ind[[h, i + 1]] rad[[i]] zog[[h, i]])/(ind[[h, i]] rad[[i]] - ind[[h, i]] zog[[h, i]] + ind[[h, i + 1]] zog[[h, i]])];
			  zog1[[h, i]] = Which[rad[[i]] =!= Infinity && zog[[h, i]] =!= -Infinity && test =!= ComplexInfinity, 
								   test, 
								   rad[[i]] =!= Infinity && zog[[h, i]] =!= -Infinity && test === ComplexInfinity, 
								   -Infinity, 
								   rad[[i]] === Infinity && zog[[h, i]] =!= -Infinity, 
								   (ind[[h, i + 1]] zog[[h, i]])/ind[[h, i]], 
								   rad[[i]] =!= Infinity && zog[[h, i]] === -Infinity, 
								   (ind[[h, i + 1]] rad[[i]] )/(-ind[[h, i]] + ind[[h, i + 1]]), 
								   rad[[i]] === Infinity && zog[[h, i]] === -Infinity, 
								   -Infinity];
			  AppendTo[zog[[h]], zog1[[h, i]] - Append[thick, 0][[i]]], {i, n}, {h, m}];
 		
		(* Focal length *)
		If[dobject === -Infinity, 
		   (* zoginf and zog1inf are the same as zog and zog1 *)
		   zoginf = zog;
		   zog1inf = zog1, 
		   
		   zoginf = ConstantArray[-Infinity, {m, 1}];
		   zog1inf = ConstantArray[0, {m, n}];
		   Table[
		         test = Quiet@Simplify[(ind[[h, i + 1]] rad[[i]] zoginf[[h, i]])/(ind[[h, i]] rad[[i]] - ind[[h, i]] zoginf[[h, i]] + ind[[h, i + 1]] zoginf[[h, i]])];
				 zog1inf[[h, i]] = Which[rad[[i]] =!= Infinity && zoginf[[h, i]] =!= -Infinity && test =!= ComplexInfinity, 
										 test, 
										 rad[[i]] =!= Infinity && zoginf[[h, i]] =!= -Infinity && test === ComplexInfinity, 
										 -Infinity, 
										 rad[[i]] === Infinity && zoginf[[h, i]] =!= -Infinity, 
										 (ind[[h, i + 1]] zoginf[[h, i]])/ind[[h, i]], 
										 rad[[i]] =!= Infinity && zoginf[[h, i]] === -Infinity, 
										 (ind[[h, i + 1]] rad[[i]] )/(-ind[[h, i]] + ind[[h, i + 1]]), 
										 rad[[i]] === Infinity && zoginf[[h, i]] === -Infinity, 
										 -Infinity];
				 AppendTo[zoginf[[h]], zog1inf[[h, i]] - Append[thick, 0][[i]]], {i, n}, {h, m}]];
		foc = zog1inf[[All, n]] - z1p;
		
		(* Height of image *)
		hobj1 = Partition[If[dobject === -Infinity, (ind[[All, 1]] zog1[[All, 1]])/ind[[All, 2]] \[Pi]/180. angle, (ind[[All, 1]] zog1[[All, 1]])/(ind[[All, 2]]zog[[All, 1]]) hobject], 1];
		Table[AppendTo[hobj1[[h]], (ind[[h, i]] zog1[[h, i]])/(ind[[h, i + 1]]zog[[h, i]]) hobj1[[h, i - 1]]], {h, m}, {i, 2, n}];
		
		(* Aberrations *)
		C1 = Table[If[Length[costasf[[i]]] === 2,
						(* the surface i is aspherical surface, so costasf = {a4, asf} where a4 can be a number or a symbol *)
						(* this formula uses costasf[[i, 1]] that is a4 (a symbol or a numerical value *)
					  -costasf[[i, 1]] * (ind[[h, i + 1]] - ind[[h, i]]) - ind[[h, i]]^2/8 * (1/(ind[[h, i + 1]] * zog1[[h, i]]) - 1/(ind[[h, i]] * zog[[h, i]])) * (1/zog[[h, i]] - 1/rad[[i]])^2, 
					  (* the surface i is conical or spherical surface, so costasf = K (in the first case - conic) or 0 (spherical) where K can be a number or a symbol *)
					  (* this formula uses costasf[[i]] that is K or 0 *)
					  -costasf[[i]] * (ind[[h, i + 1]] - ind[[h, i]])/(8rad[[i]]^3) - ind[[h, i]]^2/8 * (1/(ind[[h, i + 1]] * zog1[[h, i]]) - 1/(ind[[h, i]] * zog[[h, i]])) * (1/zog[[h, i]] - 1/rad[[i]])^2], {h, m}, {i, n}];
		(* Coefficients of coma for stop on the surface *)
		C2 = Table[ind[[h, i]]^2/2 (1/(ind[[h, i + 1]] * zog1[[h, i]]) - 1/(ind[[h, i]] * zog[[h, i]])) * (1/zog[[h, i]] - 1/rad[[i]]), {h, m}, {i, n}];
		(* alternative method using Map Map[Most, ind]^2/2 (1/(Map[Rest, ind]*zog1) - 1/(Map[Most, ind*zog]))*(1/Map[Most, zog] - 1/rad)) *)
		(* Coefficients of fields curvature and astigmatism for stop on the surface *)
		C3 = Table[ind[[h, i]]/4 * (ind[[h, i + 1]]^2 - ind[[h, i]]^2)/ind[[h, i + 1]]^2 * (1/zog[[h, i]] - 1/rad[[i]]), {h, m}, {i, n}];
		C4 = Table[-ind[[h, i]]^2/2 * (1/(ind[[h, i + 1]] * zog1[[h, i]]) - 1/(ind[[h, i]] * zog[[h, i]])) + C3[[h, i]], {h, m}, {i, n}];
		(* Coefficients of distorsion for stop on the surface *)
		C5 = Table[-(ind[[h, i]]/2)*(ind[[h, i + 1]]^2 - ind[[h, i]]^2)/ind[[h, i + 1]]^2, {h, m}, {i, n}];
		(* Aberration Coefficients depending on the stop position *)
		c1 = d1 = k1 = Cd1 = Cd2 = Cd3 = Cd4 = Cd5 = ConstantArray[0, {m, n}];
		Table[
			  {c1[[h, i]], d1[[h, i]], k1[[h, i]]} = 
			  If[zog[[h, i]] === -Infinity, {1, 0, zee[[h, i]]}, {zog[[h, i]]/(zog[[h, i]]-zee[[h, i]]), zee[[h, i]]/(zog[[h, i]]-zee[[h, i]]), zog[[h, i]] zee[[h, i]]/(zog[[h, i]]-zee[[h, i]])}];
			  Cd1[[h, i]] = c1[[h, i]]^4 C1[[h, i]];
			  Cd2[[h, i]] = c1[[h, i]]^2 (C2[[h, i]]-4k1[[h, i]]C1[[h, i]]);
			  Cd3[[h, i]] = (C3[[h, i]]-k1[[h, i]]C2[[h, i]]+2k1[[h, i]]^2 C1[[h, i]]);
			  Cd4[[h, i]] = (C4[[h, i]]-3k1[[h, i]]C2[[h, i]]+6k1[[h, i]]^2 C1[[h, i]]);Cd5[[h, i]] = 1/c1[[h, i]]^2 (C5[[h, i]]-2k1[[h, i]]C4[[h, i]]+3k1[[h, i]]^2 C2[[h, i]]-4k1[[h, i]]^3 C1[[h, i]]), {h, m}, {i, n}];
		(* Compound system *)
		(* Field angle *)
		theta = If[dobject === -Infinity, ConstantArray[\[Pi]/180. angle, m], hobject/(dobject - zee[[All, 1]])];
		(* Total spherical aberration *)
		(* Third-order spherical aberration *)
		Absfer = Table[If[i === 1, 4 (zog1[[h, n]] - zee1[[h, n]])/ind[[h, n + 1]] 1/Mae[[h, n]] Cd1[[h, 1]] * yee[[h, 1]]^3, 4 (zog1[[h, n]] - zee1[[h, n]])/ind[[h, n+1]] 1/Mae[[h, n]] Cd1[[h, i]]Mae[[h, i - 1]]^4 yee[[h, 1]]^3], {h, 1, m}, {i, 1, n}];
		Seidsphe = Table[Cd1[[h, 1]] + Sum[Cd1[[h, i]] * Mae[[h, i - 1]]^4, {i, 2, n}], {h, m}];
		Absftot = Map[Total, Absfer];
		(* Total sagittal coma *)
		Comapar = Partition[(zog1[[All, n]] - zee1[[All, n]])/ind[[All, n + 1]] 1/Mae[[All, n]] Cd2[[All, 1]] theta[[1]](yee[[All, 1]])^2, 1];
		Table[AppendTo[Comapar[[h]], (zog1[[h, n]]-zee1[[h, n]])/ind[[h, n+1]] 1/Mae[[h, n]] ind[[h, 1]]/ind[[h, i]] (Mae[[h, i - 1]])^2 Cd2[[h, i]] theta[[h]](yee[[h, 1]])^2], {h, m}, {i, 2, n}];
		Seidcoma = Table[Cd2[[h, 1]]+Sum[ind[[h, 1]]/ind[[h, i]] (Mae[[h, i - 1]])^2 Cd2[[h, i]], {i, 2, n}], {h, m}];
		Comatot = Map[Total, Comapar];
		(* Astigmatism *)
		Astpar = Partition[(zog1[[All, n]]-zee1[[All, n]])/ind[[All, n+1]] 1/Mae[[All, n]] (Cd4[[All, 1]]-Cd3[[All, 1]]) theta^2yee[[All, 1]], 1];
		Table[AppendTo[Astpar[[h]], (zog1[[h, n]]-zee1[[h, n]])/ind[[h, n+1]] 1/Mae[[h, n]] (ind[[h, 1]]/ind[[h, i]])^2 (Cd4[[h, i]]-Cd3[[h, i]]) theta[[h]]^2yee[[h, 1]]], {h, m}, {i, 2, n}];
		Seidast = Map[Total, (ind[[All, 1]]/Map[Most, ind])^2 (Cd4-Cd3)];
		Asttot = Map[Total, Astpar];
		Sp = ind[[All, n+1]]/ConstantArray[rad, m] (1/Map[Rest, ind]-1/Map[Most, ind]);
		Seidcurv = Map[Total, ind[[All, 1]]^2/Mae[[All, n]] Sp/2];
		
		(* non funziona in tutti i casi Curvpar = Transpose[(zog1[[All, n]]-zee1[[All, n]])/Map[Last, ind]^2 ind[[All, 1]]^2/Mae[[All, n]] Sp/2 theta^2 yee[[All, 1]]]; *)
		
		Curvpar = Table[(zog1[[h, n]] - zee1[[h, n]])/ind[[h, n + 1]]^2 ind[[h, 1]]^2/Mae[[h, n]] Sp[[h, i]]/2 theta[[h]]^2 yee[[h, 1]], {h, 1, m}, {i, 1, n}];
		
		
		Curvtot = Map[Total, Curvpar];
		Petzval = 1/Map[Total, Sp];
		(* Distorsion *)
		Distpar = Partition[(zog1[[All, n]]-zee1[[All, n]])/ind[[All, n+1]] 1/Mae[[All, n]] Cd5[[All, 1]] theta^3, 1];
		Table[AppendTo[Distpar[[h]], (zog1[[h, n]]-zee1[[h, n]])/ind[[h, n+1]] 1/Mae[[h, n]] (ind[[h, 1]]/ind[[h, i]])^3 Cd5[[h, i]] 1/Mae[[h, i - 1]]^2 theta[[h]]^3], {h, m}, {i, 2, n}];
		Seiddist = 1/Mae[[All, n]] Cd5[[All, 1]]+Map[Total, 1/Mae[[All, n]] (ind[[All, 1]]/ind[[All, 2;;-2]])^3 Map[Rest, Cd5] 1/Map[Most, Mae]^2];
		Disttot = (zog1[[All, n]]-zee1[[All, n]])/ind[[All, n+1]] Seiddist*theta^3;
	    

		(* exported variables are always set *)
		{GOn = n,
		 GOwe = de[[1]],
		 GOren = If[Depth[diame] === 2, diame[[1]], diame[[1, 1]]],
		 GOwen = zee1[[1, n]],
		 GOrexit = Abs[yee1[[1, n]]],
		 GOdistancefp = zp[[1]],
		 GOdistancesp = z1p[[1]],
		 GOdistancegauss = zog1,
		 GOfocalLength = foc,
		 GOimageHeight = hobj1,
		 GOSphericalCoefficient = TimeConstrained[Simplify[Seidsphe[[1]]], $SimplifyTimeLimit, Seidsphe[[1]]],
		 GOComaCoefficient = TimeConstrained[Simplify[Seidcoma[[1]]], $SimplifyTimeLimit, Seidcoma[[1]]],
		 GOAstigmatismCoefficient = TimeConstrained[Simplify[Seidast[[1]]], $SimplifyTimeLimit, Seidast[[1]]],
		 GOaberration = TimeConstrained[Simplify[Absftot], $SimplifyTimeLimit, Absftot],
		 GOcoma = TimeConstrained[Simplify[Comatot], $SimplifyTimeLimit, Comatot],
		 GOastigmatism = TimeConstrained[Simplify[Asttot], $SimplifyTimeLimit, Asttot],
		 GOcurvature = TimeConstrained[Simplify[Curvtot[[1]]], $SimplifyTimeLimit, Curvtot[[1]]],
		 GOSeidelCurvature = TimeConstrained[Simplify[Seidcurv[[1]]], $SimplifyTimeLimit, Seidcurv[[1]]],
		 GOPetzvalRadius = Petzval[[1]],
		 GOdistortion = TimeConstrained[Simplify[Disttot[[1]]], $SimplifyTimeLimit, Disttot[[1]]]
		};
		
  (* defines the two panels, input parameters and output values *)
  
  inPanel = Grid[{	{"Radius of S" , "rad", Row[rad, ", "]},
					{"Thickness of corrector", "thick", Row[thick, ", "]},
					{"Refractive indices", "ind", Column[Map[Row[#, ", "] &, ind]]},
					{"Conic constant", "costasf", Row[costasf, ", "]},
					{"Stop radius", "stoprad", stoprad},
					{"Number of surfaces before stop", "nstop", nstop},
					{"Distance between corrector and mirror", "dis", dis},
					{"Distance of the object", "dobject", dobject},
					{"Height of the object", "hobject", hobject},
					{"View angle", "angle", angle},
					{"List of wave lengths", "waves", Row[waves, ", "]}}, 
					Alignment -> {{Left, Left, Right}, Center}, 
					Spacings -> {2, 1}, 
					Dividers -> Center, 
					FrameStyle -> LightGray,
					BaseStyle->{"InputParameterBottom"}];
  
  reportWaves = MapThread[If[NumericQ[#1], Row[{Subscript["\[Lambda]", #2], " = ", #1}], Subscript["\[Lambda]", #2]] &, {waves, Range[Length[waves]]}];
  outPanel = If[OptionValue[OutputLevel] === "Coefficients",
				Grid[{{"Spherical coefficient", "GOSphericalCoefficient", GOSphericalCoefficient},
					{"Coma coefficient", "GOComaCoefficient", GOComaCoefficient},
					{"Astigmatism coefficient", "GOAstigmatismCoefficient", GOAstigmatismCoefficient}
					}, 
					Alignment -> {{Left, Right}, Center}, 
					Spacings -> {2, 1}, 
					Dividers -> Center, 
					FrameStyle -> LightGray,
					BaseStyle->{"OutputValueBottom"}],
				Grid[{{Row[{"Distance of entrance pupil from the first surface for ", First@reportWaves}], Subscript["w", "e"], GOwe},
					{Row[{"Radius of the entrance pupil for ", First@reportWaves}], Subscript["r", "en"], GOren},
					{Row[{"Distance of the exit pupil from the last surface for ", First@reportWaves}], Subscript["w", "en"], GOwen},
					{Row[{"Radius of the exit pupil for ", First@reportWaves}], Subscript["r", "ex"], GOrexit},
					{"Distance of the first principal plane from the first surface", "", GOdistancefp},
					{"Distance of the second principal plane from the last surface", "", GOdistancesp},
					{Row[{"Gaussian distance of the images from the surface for ", First@reportWaves}], "", 
							Column[Table[GOdistancegauss[[1, i]], {i, n}], Alignment -> Right, Spacings -> 1, BaseStyle -> "Pane"]},
					{"Focal length", "", GOfocalLength[[1]]},
					{"Image height", "", GOimageHeight[[1, n]]},
					{"Third-order spherical aberration", Column[waves], Column[GOaberration]},
					{"Third-order coma", Column[waves], Column[GOcoma]},
					{"Third-order astigmatism", Column[waves], Column[GOastigmatism]},
					{"Third-order curvature", "", GOcurvature},
					{"Petzval radius", "", GOPetzvalRadius},
					{"Third-order distortion", "", GOdistortion},
					{"Spherical coefficient", "", GOSphericalCoefficient},
					{"Coma coefficient", "", GOComaCoefficient},
					{"Astigmatism coefficient", "", GOAstigmatismCoefficient}				
					}, 
					Alignment -> {{Left, Right}, Center}, 
					Spacings -> {2, 1}, 
					Dividers -> Center, 
					FrameStyle -> LightGray,
					BaseStyle->{"OutputValueBottom"}]];
				
  (* generates the type of output required *)
  Switch[OptionValue[OutputType],
		 "Values",
		 Return[{GOwe, GOren, GOwen, GOrexit, GOdistancefp, GOdistancesp, GOdistancegauss, GOfocalLength, GOimageHeight, GOaberration, GOcoma, GOastigmatism, 
				 GOcurvature, GOPetzvalRadius, GOdistortion, GOSphericalCoefficient, GOComaCoefficient, GOAstigmatismCoefficient}],
		 
		 "Report",
		 GenerateDocument[TemplateApply[$ReportTemplate, 
										Join[<|	"title" -> $GeometricOpticsPackagesList[SelectFirst[#PackageName == "TotalAberrations" &], "Description"], 
												"date" -> DateString[], 
												"function" -> "TotalAberrations", 
												"outputlevel" -> OptionValue[OutputLevel],
												"inPanel" -> inPanel, 
												"outPanel" -> outPanel |>]]];,

		 "Print",
		 CellPrint[TextCell[TemplateApply[$PrintTemplate, 
											Join[<|	"title" -> $GeometricOpticsPackagesList[SelectFirst[#PackageName == "TotalAberrations" &], "Description"], 
													"date" -> DateString[], 
													"function" -> "TotalAberrations", 
													"outputlevel" -> OptionValue[OutputLevel],
													"inPanel" -> inPanel, 
													"outPanel" -> outPanel |>]], "Text"]];,
		 "Basic",
		 Return[TemplateApply[$BasicTemplate, 
								Join[<| "outputlevel" -> OptionValue[OutputLevel],
									"inputs" -> {{"rad", rad}, 
												{"thick", thick},
												{"ind", ind},
												{"costasf", costasf},
												{"stoprad", stoprad},
												{"nstop", nstop},
												{"dis", dis},
												{"dobject", dobject},
												{"hobject", hobject},
												{"angle", angle},
												{"waves", waves}},
									If[OptionValue[OutputLevel] === "Coefficients",
										"outputs" -> {	{"GOSphericalCoefficient", GOSphericalCoefficient},
														{"GOComaCoefficient", GOComaCoefficient},
														{"GOAstigmatismCoefficient", GOAstigmatismCoefficient}},
										"outputs" ->{	{"GOwe", GOwe},
														{"GOren", GOren}, 
														{"GOwen", GOwen},
														{"GOrexit", GOrexit}, 
														{"GOdistancefp", GOdistancefp},
														{"GOdistancesp", GOdistancesp},
														{"GOdistancegauss", GOdistancegauss},
														{"GOfocalLength", GOfocalLength},
														{"GOimageHeight", GOimageHeight},
														{"GOaberration", GOaberration},
														{"GOcoma", GOcoma},
														{"GOastigmatism", GOastigmatism},
														{"GOcurvature", GOcurvature},
														{"GOPetzvalRadius", GOPetzvalRadius},
														{"GOdistortion", GOdistortion}
													}]|>]]],
		 "Test",
		 {n, m, rade, inde1, ze, ze1, de, ye, diame, ye1, zee, zee1, fe, yee, yee1, Mae, P, R0, Mt, D0, B0, b, zp, z1p, zog, zog1, test, zoginf, zog1inf, 
		foc, hobj1, C1, C2, C3, C4, C5, c1, d1, k1, Cd1, Cd2, Cd3, Cd4, Cd5, theta, mhobject, Absfer, Absftot, Comapar, Comatot, Asttot, Astpar, Sp, 
		Seidcurv, Curvpar, Curvtot, Petzval, Seidsphe, Seidcoma, Seidast, Seiddist, Disttot, Distpar, h},
		 
		 (* GOs variables are set but not shown *)
		 _,
		 Return[]
		]];


LoadPackage[packageName_String] := If[MemberQ[$InstalledPackages[All, "PackageName"], packageName], Get["GeometricOptics`Packages`" <> packageName <> "`"], 
										MessageDialog[Row[{"The required package ", Style[packageName, Bold], " cannot be loaded because it's not installed"}], WindowTitle -> "Warning: package not installed"]];
		
fileOpenerIcon = Import[FileNameJoin[{$IconsDirectory, "FileOpener.png"}]];
selectAllIcon = Import[FileNameJoin[{$IconsDirectory, "SelectAll.png"}]];
clearAllIcon = Import[FileNameJoin[{$IconsDirectory, "ClearAll.png"}]];
packageLoaderIcon = Import[FileNameJoin[{$IconsDirectory, "PackageLoader.png"}]];
checkInstallationIcon = Import[FileNameJoin[{$IconsDirectory, "CheckInstallation.png"}]];
folderOpenIcon = Import[FileNameJoin[{$IconsDirectory, "FolderOpen.png"}]];
mainIcon = Import[FileNameJoin[{$IconsDirectory, "GeometricOptics.png"}]];
helpIcon = Import[FileNameJoin[{$IconsDirectory, "help.png"}]];
saveLastIcon = Import[FileNameJoin[{$IconsDirectory, "SaveLast.png"}]];
saveFromListIcon = Import[FileNameJoin[{$IconsDirectory, "SaveFromList.png"}]];

GenerateMainPalette[]:=
Module[{selectedPackages},
	CreatePalette[
			DocumentNotebook[{Row[{
				Tooltip[Button[mainIcon, 
								(Get["GeometricOptics`"];
								 Map[LoadPackage, Normal[$InstalledPackages[All, "PackageName"]]]), 
							   Method -> "Queued", 
							   Appearance -> "Frameless", 
							   ImageSize -> {50, 50}], 
						"Load all installed packages"],
				
				Tooltip[Button[packageLoaderIcon, 
								(Needs["GeometricOptics`"];
								selectedPackages = {};
								CreatePalette[{Column[{	Spacer[3],TextCell["Select one or more package to load"], Spacer[3],
														Row[{Button[Tooltip[selectAllIcon, "Select all packages"], 
																	selectedPackages = Normal[$InstalledPackages[All, "PackageName"]], 
																	ImageSize -> {30, 30}, 
																	Appearance -> "Framed"], 
														Spacer[10], 
														Button[Tooltip[clearAllIcon, "Clear current selection"], 
																selectedPackages = {}, 
																ImageSize -> {30, 30}, 
																Appearance -> "Framed"]}], 
														Spacer[3],
														ListPicker[Dynamic[selectedPackages], 
																	Thread[Rule[Normal[$InstalledPackages[All, "PackageName"]], 
																				Map[StringJoin[Riffle[#, " "]]&, 
																					StringSplit[Normal[$InstalledPackages[All, "PackageName"]], RegularExpression["(?=[[:upper:]])"]]]]]], 
														Spacer[5],
														Row[{Button[Tooltip["Load", "Load only selected packages"],
																	(Map[LoadPackage, selectedPackages]; 
																	 selectedPackages = {}; 
																	 NotebookClose[ButtonNotebook[]]), 
																	Method -> "Queued"], 
															Spacer[2],
															Button[Tooltip["Close", "Close without to load packages"], 
																	(selectedPackages = {}; 
																	 NotebookClose[ButtonNotebook[]])]}]}, 
														Alignment -> Center]}, 
												WindowMargins -> Automatic, 
												TextAlignment -> Center, 
												WindowSize -> {250, 400}, 
												WindowTitle -> "Package loader"]),
								Method -> "Queued", 
								Appearance -> "Frameless", 
								ImageSize -> {50, 50}], "Choose packages to load"],
				Tooltip[Button[fileOpenerIcon,
					(Needs["GeometricOptics`"];
					 selectedPackages = {};
					 CreatePalette[{Column[{Spacer[3],TextCell["Select one or more file to open"], Spacer[3],
											Row[{Button[Tooltip[selectAllIcon, "Select all files"], selectedPackages = Normal[$InstalledPackages[All, "PackageName"]], ImageSize -> {30, 30}, Appearance -> "Framed"], Spacer[10], 
												 Button[Tooltip[clearAllIcon, "Clear current selection"], selectedPackages = {}, ImageSize -> {30, 30}, Appearance -> "Framed"]}], Spacer[3],
											ListPicker[Dynamic[selectedPackages], 
														Thread[Rule[Normal[$InstalledPackages[All, "PackageName"]], 
																	Map[StringJoin[Riffle[#, " "]] &, StringSplit[Normal[$InstalledPackages[All, "PackageName"]], RegularExpression["(?=[[:upper:]])"]]]]]], Spacer[5],
											Row[{Button[Tooltip["Open", "Open main file from selected packages"],
														(Map[NotebookOpen[FileNameJoin[{$PackagesDirectory, #, # <> ".nb"}]] &, selectedPackages];
														 selectedPackages = {};
														 NotebookClose[ButtonNotebook[]]), Method -> "Queued"],  Spacer[2],
												 Button[Tooltip["Close", "Close without to load packages"],
														(selectedPackages = {}; 
														 NotebookClose[ButtonNotebook[]])]}]}, Alignment -> Center]}, WindowMargins -> Automatic, TextAlignment -> Center, WindowSize -> {250, 400}, WindowTitle -> "File opener"]),
					Method -> "Queued", Appearance -> "Frameless", ImageSize -> {50, 50}], "Package's file opener"],
				Tooltip[Button[checkInstallationIcon, 
								(Needs["GeometricOptics`"];
								 CheckInstalledPackages[];), Method -> "Queued", ImageSize -> {50, 50}, Appearance -> "Frameless"], "Check installed package from Geometric Optics book"],
				Tooltip[Button[folderOpenIcon, SystemOpen[$MainDirectory], Method -> "Queued", Appearance -> "Frameless", ImageSize -> {50, 50}], "Open GeometricOptics main folder"],
				
				Tooltip[Button[saveLastIcon, SaveLastExample[], Method -> "Queued", ImageSize -> {50, 50}, Appearance -> "Frameless"], "Save last example calculated"],
				
				Tooltip[Button[saveFromListIcon, SaveExampleFromList[], Method -> "Queued", Appearance -> "Frameless", ImageSize -> {50, 50}], "Save an example from the stack of current session's examples"]
				
				}, Spacer[15]]},
				
				Background -> White, CellMargins -> {{20, 20}, {10, 10}}, WindowTitle -> "Geometric Optics main palette", WindowSize -> All]];];


ArgumentsQ[Package_String, Arguments_Association]:=
 Module[{types, values, type},
		types = Normal@Values@$GeometricOpticsPackagesList[SelectFirst[#PackageName == Package &], "Arguments"][All, "Type"];
		(* values takes values of variables from Arguments, passing through the definition of Arguments of Package, so to check also the correctness of number and the names of variables *)
		values = Normal@ReplaceAll[Keys@$GeometricOpticsPackagesList[SelectFirst[#PackageName == Package &], "Arguments"], Arguments];
		Apply[And, MapThread[#1[#2]&, {Map[Function[type, $ArgumentTypes[SelectFirst[#Type === type &], "CheckFunction"]], types], values}]]]/;MemberQ[$InstalledPackages, Package, Infinity];

ArgumentsQ[Package_String, Arguments:{{_, _}..}]:= ArgumentsQ[Package, AssociationThread[Apply[Sequence, Transpose@Arguments]]];

ArgumentsQ[Package_String, Arguments:{{_, _ ...}, {_, _ ...}}]:= ArgumentsQ[Package, AssociationThread[Apply[Sequence, Arguments]]]/;MatchQ[Dimensions[Arguments], {2, _}];

ArgumentsQ[Package_String, Arguments:Rule[{_, _ ...}, {_, _ ...}]]:= ArgumentsQ[Package, AssociationThread[Arguments]]/;MatchQ[Dimensions[Apply[List, Arguments]], {2, _}];

ArgumentsQ[___]:=False;

(* $ExamplesArchive has duplicates which may have different ExampleID and Description but are the same for input parameters/function *)
(* This function can be applied to $ExamplesArchive or any other ExampleList like those related to specific package *)
(* This doesn't not set any variable or save any file, just clear the given list *)
DeleteExampleDuplicates[ExampleList_]:= 
 Module[{position},
		(* vars keeping examples have different structure, so this finds the three keys on which base the deleteduplicate *)
		position = Position[First@Normal@ExampleList, Alternatives["PackageID", "PackageName", "Arguments"], Infinity][[All, 1]];
		Map[AssociationThread[Apply[Rule, #]] &, Map[Transpose, DeleteDuplicatesBy[Normal@Map[KeyValueMap[List, #] &, ExampleList], Part[#, position] &]]]];

(* Save the latest computed example, the last item in the $ExamplesStack *)				
SaveLastExample[]:=
 Module[{example, arguments, description = ""},
		If[$ExamplesStack =!= {},
		   example = Last[$ExamplesStack];
		   arguments = {Keys[example["Arguments"]], Values[example["Arguments"]]};
		   If[DialogInput[DialogNotebook[
										Panel[Column[{Style["Would you like to save the latest example with the following parameters?", "Text"], 
														Spacer[5],
														Grid[{	{"Package", NameFromPackageName[Key["PackageName"][example]]}, 
																{"Function", Key["PackageName"][example]},
																{"Input parameters", Grid[Transpose[arguments], Alignment -> Left]}}, 
															Alignment -> Left, 
															Spacings -> {1, 1}],
														Spacer[5],
														Column[{"Add a short description", InputField[Dynamic[description], String, FieldHint -> "Leave empty for None"]}],
														Row[{Button["Save", DialogReturn[True]], Button["Cancel", DialogReturn[False]]}]},
													Alignment -> Center]],
										WindowTitle -> "Save latest calculated example"
										]],
				SaveExample[Key["PackageName"][example], description, arguments]],
			MessageDialog["There are no computed examples in the current working session.", WindowTitle->"Warning: example not saved"]]];

(* Let the user select an example to save from the $ExamplesStack *)				
SaveExampleFromList[]:=
 DynamicModule[{packages, arguments, examples, example, save, description = ""},
				If[$ExamplesStack =!= {}, 
					packages = Lookup[$ExamplesStack, "PackageName" ];
					arguments = Lookup[$ExamplesStack, "Arguments" ];
					examples = MapThread[Framed[Grid[Prepend[Transpose[KeyValueMap[List, Association[#2]]], {Style[#1, Bold], Apply[Sequence, ConstantArray[SpanFromLeft, Length[#2] - 1]]}], 
													 Alignment -> Center, Spacings -> {1, 1}, Dividers -> Center], FrameStyle -> LightGray] &, {packages, arguments}];
					example = 1;
					save = DialogInput[DialogNotebook[Panel[Column[{Style["Choose the example to save from the example stack of the current working session", "Text"], 	
																	Spacer[5],
																	PopupMenu[Dynamic[example], Thread[Rule[Range[Length[examples]], examples]], FrameMargins -> 5, Alignment -> Center],
																	Spacer[5],
																	Column[{"Add a short description", InputField[Dynamic[description], String, FieldHint -> "Leave empty for None"]}], 
																	Row[{Button["Save", DialogReturn[True]], Button["Cancel", DialogReturn[False]]}]}, 
																	Alignment -> Center]],
														WindowTitle -> "Choose an example from the stack"]];
					If[save === True,
						SaveExample[packages[[example]], description, Association[arguments[[example]]]]],
					(* there are no examples in $ExamplesStack *)
					MessageDialog["There are no computed examples in the current working session.", WindowTitle -> "Warning: example not saved"]]];
			
(* SaveExample saves an example if all arguments are correct, otherwise issue an error message *)
SaveExample[Package_String, Description_String, Arguments_Association]:=
 Module[{args, defaultvalues, description, values, newexample, counter},
		Which[
				FreeQ[$PackageNames, Package],
				MessageDialog["Example not created, the package " <> Package <> " doesn't exist.", WindowTitle -> "Warning: example not saved"],
				
				FreeQ[$InstalledPackages, Package],
				MessageDialog["Example not created, the package " <> Package <> " is not installed", WindowTitle -> "Warning: example not saved"],
				
				ArgumentsQ[Package, Arguments],
				(* arguments are correct in number and type *)
				counter = $ExamplesArchive[Max, "ExampleID"];
				newexample = Dataset[{<|"ExampleID" -> If[Not[NumericQ[counter]], 0, counter] + 1,
										"PackageID" -> $GeometricOpticsPackagesList[SelectFirst[#PackageName == Package &], "PackageID"],
										"PackageName" -> Package,
										"Arguments" -> Arguments, 
										"Description" -> Description|>}];
				(* append the new example to the general list of examples from all packages *)
				$ExamplesArchive = Join[$ExamplesArchive, newexample];
				If[Not@FileExistsQ@$ExamplesDirectory[Package],
					(* the first time creates the folder and the example.m file *)
					CreateDirectory@$ExamplesDirectory[Package];
					CreateFile@FileNameJoin[{$ExamplesDirectory[Package], "Examples.m"}]];				
				(* append the new example to the package's list of examples too *)
				Quiet@Export[FileNameJoin[{$ExamplesDirectory[Package], "Examples.m"}], 
							DeleteExampleDuplicates[Join[ReplaceAll[Import[FileNameJoin[{$ExamplesDirectory[Package], "Examples.m"}]], (Null | $Failed) -> {}], newexample]]],
			
				True,
				MessageDialog["The values given are not correct, example not saved for the package " <> Package, WindowTitle -> "Warning: example not saved"]]];

SaveExample[Package_String, Description_String, Arguments:{{_, _}..}]:= SaveExample[Package, Description, AssociationThread[Apply[Sequence, Transpose@Arguments]]];

SaveExample[Package_String, Description_String, Arguments:{{_, _ ...}, {_, _ ...}}]:= SaveExample[Package, Description, AssociationThread[Apply[Sequence, Arguments]]]/;MatchQ[Dimensions[Arguments], {2, _}];

SaveExample[Package_String, Description_String, Arguments:Rule[{_, _ ...}, {_, _ ...}]]:= SaveExample[Package, Description, AssociationThread[Arguments]]/;MatchQ[Dimensions[Apply[List, Arguments]], {2, _}];

SaveExample[Package_String, Description_String, Arguments:{Rule[_, _] ...}]:= SaveExample[Package, Description, Association[Arguments]]/;MatchQ[Dimensions[Apply[List, Arguments]], {2, _}];


LoadExample[package_String] :=
 Module[{examplesAll, examples, example},
  examplesAll = Join[Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
  examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], Alignment -> Center, Spacings -> {1, 1}, Dividers -> Center], FrameStyle -> LightGray] &, examplesAll];
  example = "Browse...";
  DialogInput[DialogNotebook[Panel@Column[{"Load an example from the archives (current session and saved DB)", 
											Row[{Dynamic@PopupMenu[Dynamic[example], Thread[Rule[Range[Length[examples]], examples]], If[examples === {}, "No example saved", "Browse..."], FrameMargins -> 3, Alignment -> Center]}],
											Row[{Button["Get value", DialogReturn[Values[examplesAll[[example]]]]], Button["Clear", DialogReturn[False]]}]}, Alignment -> Center, Spacings -> 1.5],
							WindowTitle -> "Load an example of " <> package]]];
			   
Options[CheckInstalledPackages] = {ShowResult -> True, Results -> False};
CheckInstalledPackages[OptionsPattern[]] := Module[{},
													If[OptionValue[ShowResult],
														MessageDialog[Which[Length[$InstalledPackages] > 0 && Length[$NotInstalledPackages] > 0,
																			(* there are installed packages and missing packages *)
																			Column[{"Packages installed", Spacer[5], 
																					$InstalledPackages[All, {"Category", "PackageName", "Description"}], Spacer[5], 
																					"Packages not installed", Spacer[5],
																					$NotInstalledPackages[All, {"Category", "PackageName", "Description"}], DefaultButton[]}, Alignment -> Center],
																			
																			Length[$InstalledPackages] > 0 && Length[$NotInstalledPackages] === 0,
																			(* all packages are installed *)
																			Column[{"All available packages are installed", Spacer[5], 
																					$InstalledPackages[All, {"Category", "PackageName", "Description"}], DefaultButton[]}, Alignment -> Center],
																			
																			Length[$InstalledPackages] === 0,
																			(* there aren't installed packages *)
																			Column[{"No packages are installed", Spacer[5], 
																					"Packages not installed", Spacer[5], 
																					$NotInstalledPackages[All, {"Category", "PackageName", "Description"}], DefaultButton[]}, Alignment -> Center]],
																	   WindowTitle -> "Installed Geometric Optics Packages", WindowSize -> Fit]];
													If[OptionValue[Results], 
														Return@Normal@$InstalledPackages[All, "PackageName"]]];
						 
								 
										
 End[];
 EndPackage[];

 (******* System changes *******)
Dataset`$ElisionThreshold = 200;